#include "MainWindow.h"

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    // Variable initialization
    circuit = new Circuit();

    // Zoom slider and text
    zoomValue = new QLabel;
    zoomValue->setAlignment(Qt::AlignCenter);
    zoomSlider = new QSlider;
    zoomSlider->setMinimum(0);
    zoomSlider->setMaximum(200);
    zoomSlider->setValue(100);
    zoomSlider->setTickPosition(QSlider::TicksBothSides);
    zoomSlider->setTickInterval(10);

    // Canvas
    canvas = new Canvas;

    // Zoom slider and text layout
    QVBoxLayout *zoomLayout = new QVBoxLayout();
    zoomLayout->addWidget(zoomValue);
    zoomLayout->addWidget(zoomSlider);
    canvas->zoomCanvas(100);
    updateZoomValue(100);

    // Top-level layout and widget
    QHBoxLayout *mainLayout = new QHBoxLayout();
    mainLayout->addWidget(canvas);
    mainLayout->addLayout(zoomLayout);

    QWidget *mainWidget = new QWidget();
    mainWidget->setLayout(mainLayout);
    setCentralWidget(mainWidget);

    // Connect signals
    connect(zoomSlider, SIGNAL(valueChanged(int)), canvas, SLOT(zoomCanvas(int)));
    connect(zoomSlider, SIGNAL(valueChanged(int)), this, SLOT(updateZoomValue(int)));
    connect(canvas, SIGNAL(updateZoomSlider(int)), zoomSlider, SLOT(setValue(int)));
    connect(canvas, SIGNAL(updateStatus(QString)), this, SLOT(updateStatus(QString)));
    connect(circuit, SIGNAL(updateStatus(QString)), this, SLOT(updateStatus(QString)));

    setWindowTitle("Circuit Visualizer");
    updateStatus("Waiting to load circuit file...");

    createActions();
    createMenus();
}

MainWindow::~MainWindow() {

}

void MainWindow::updateZoomValue(int value)
{
    zoomValue->setText(QString::number(value));
}

void MainWindow::updateStatus(QString string)
{
    statusBar()->showMessage(string);
}

void MainWindow::loadCircuit()
{
    static QString lastKnownFilePath = ".";

    QString file = QFileDialog::getOpenFileName(this, "Circuit (lev) File", lastKnownFilePath, "Vector File (*.lev)");
    QCoreApplication::processEvents(); // Close file dialog before proceeding

    if (file.size() == 0) return;

    // If file was selected, save directory for next time dialog is opened
    QFileInfo fileInfo = QFileInfo(file);
    lastKnownFilePath = fileInfo.absolutePath();

    if (circuit->readGates(file)) {
        statusBar()->showMessage(circuit->getLastError());
        return;
    }

    canvas->drawCircuit(circuit);
    loadCircuitFileAction->setEnabled(false);
    showSimControllerAction->setEnabled(true);
    resetCircuitAction->setEnabled(true);
}

void MainWindow::showHelp()
{
    QMessageBox msg;
    msg.setWindowTitle("Help");
    msg.setText("- Hold down CTRL to pan"
                "\n- Zoom with mouse wheel"
                "\n- Circled gates are queued");
    msg.exec();
}

void MainWindow::createActions()
{
    loadCircuitFileAction = new QAction("Load &Circuit", this);
    loadCircuitFileAction->setStatusTip("Loads a new circuit from file");
    connect(loadCircuitFileAction, SIGNAL(triggered()), this, SLOT(loadCircuit()));

    showHelpAction = new QAction("&About", this);
    showHelpAction->setStatusTip("Instructions on program usage");
    connect(showHelpAction, SIGNAL(triggered()), this, SLOT(showHelp()));

    resetCircuitAction = new QAction("&Reset Circuit", this);
    resetCircuitAction->setStatusTip("Resets circuit to default values");
    resetCircuitAction->setEnabled(false);
    connect(resetCircuitAction, SIGNAL(triggered()), circuit, SLOT(reset()));

    showSimControllerAction = new QAction("&Begin Simulator", this);
    showSimControllerAction->setStatusTip("Shows simulation controller");
    showSimControllerAction->setEnabled(false);
    connect(showSimControllerAction, SIGNAL(triggered()), circuit, SLOT(showSimController()));

    toggleWireValuesAction = new QAction("Toggle &Wire Values", this);
    toggleWireValuesAction->setStatusTip("Show/hide values on wires");
    connect(toggleWireValuesAction, SIGNAL(triggered()), circuit, SIGNAL(toggleShowWireValues()));
}

void MainWindow::createMenus()
{
    fileMenu = menuBar()->addMenu("&File");
    fileMenu->addAction(loadCircuitFileAction);

    circuitMenu = menuBar()->addMenu("&Circuit");
    circuitMenu->addAction(resetCircuitAction);
    circuitMenu->addAction(showSimControllerAction);
    circuitMenu->addAction(toggleWireValuesAction);

    helpMenu = menuBar()->addMenu("&Help");
    helpMenu->addAction(showHelpAction);
}
